import uuid
from django.db import models
from django.utils import timezone
from app.models import ClassLevel, Subject, StudentProfile

# === Annual Subscription Models ===
class AnnualSubscriptionDetails(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    title = models.CharField(max_length=255, blank=True, null=True)
    description = models.TextField(blank=True, null=True)
    class_level = models.ForeignKey(ClassLevel, on_delete=models.CASCADE, blank=True, null=True)
    subject = models.ForeignKey(Subject, on_delete=models.CASCADE, related_name='annual_subscription_subjects', blank=True, null=True)
    price = models.IntegerField(blank=True, null=True, default=0)
    image = models.ImageField(upload_to='subscription/annual_img', blank=True, null=True)
    
    def __str__(self):
        return self.title or ""

class AnnualSubscriptionkeys(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    details = models.ForeignKey(AnnualSubscriptionDetails, on_delete=models.CASCADE, related_name='keys', blank=True, null=True)
    key = models.CharField(max_length=255, unique=True, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True, blank=True, null=True)

    def __str__(self):
        return self.key or ""

class AnnualSubscription(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    annual_subscription_details = models.ForeignKey(AnnualSubscriptionDetails, on_delete=models.CASCADE, related_name='subscriptions', blank=True, null=True)
    student = models.ForeignKey(StudentProfile, on_delete=models.CASCADE, related_name='annual_subscriptions', blank=True, null=True)
    class_level = models.ForeignKey(ClassLevel, on_delete=models.CASCADE, blank=True, null=True)
    start_date = models.DateField(auto_now_add=True, blank=True, null=True)
    end_date = models.DateField(blank=True, null=True)
    is_active = models.BooleanField(default=True)

    def is_expired(self):
        return timezone.now().date() > self.end_date if self.end_date else False

    def days_remaining(self):
        return (self.end_date - timezone.now().date()).days if self.end_date else 0

    def __str__(self):
        return f"{self.student.user.username if self.student else 'N/A'} - Annual - {self.class_level.name if self.class_level else 'N/A'}"

# === Subject-Based Subscription Models ===
class SubjectBasedSubscriptionDetails(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    title = models.CharField(max_length=255, blank=True, null=True, default="")
    description = models.TextField(blank=True, null=True, default="")
    class_level = models.ForeignKey(ClassLevel, on_delete=models.CASCADE, blank=True, null=True)
    subject = models.ForeignKey(Subject, on_delete=models.CASCADE, related_name='subject_based_subscription_subjects', blank=True, null=True)
    price = models.IntegerField(blank=True, null=True, default=0)
    image = models.ImageField(upload_to='subscription/subject_img', blank=True, null=True)

    def __str__(self):
        return self.title or ""

class SubjectBasedSubscriptionkeys(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    details = models.ForeignKey(SubjectBasedSubscriptionDetails, on_delete=models.CASCADE, related_name='keys', blank=True, null=True)
    key = models.CharField(max_length=255, unique=True, blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True, blank=True, null=True)

    def __str__(self):
        return self.key or ""

class SubjectBasedSubscription(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    subject_based_subscription_details = models.ForeignKey(SubjectBasedSubscriptionDetails, on_delete=models.CASCADE, related_name='subscriptions', blank=True, null=True)
    student = models.ForeignKey(StudentProfile, on_delete=models.CASCADE, related_name='subject_subscriptions', blank=True, null=True)
    class_level = models.ForeignKey(ClassLevel, on_delete=models.CASCADE, blank=True, null=True)
    subjects = models.ManyToManyField(Subject, blank=True)
    start_date = models.DateField(auto_now_add=True, blank=True, null=True)
    end_date = models.DateField(blank=True, null=True)
    is_active = models.BooleanField(default=True)

    def is_expired(self):
        return timezone.now().date() > self.end_date if self.end_date else False

    def days_remaining(self):
        return (self.end_date - timezone.now().date()).days if self.end_date else 0

    def __str__(self):
        return f"{self.student.user.username if self.student else 'N/A'} - Subject-Based - {self.class_level.name if self.class_level else 'N/A'}"





# === Bundle Subscription Models ===
class BundleSubscriptionDetails(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    title = models.CharField(max_length=255, blank=True, null=True, default="")
    description = models.TextField(blank=True, null=True, default="")
    class_level = models.ForeignKey(ClassLevel, on_delete=models.CASCADE, blank=True, null=True)
    subjects = models.ManyToManyField(Subject, blank=True, related_name='bundle_subscription_details')
    price = models.IntegerField(blank=True, null=True, default=0)
    image = models.ImageField(upload_to='subscription/bundle_img', blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def __str__(self):
        return self.title or ""

    def get_subjects_list(self):
        return ", ".join([subject.name for subject in self.subjects.all()])
    
    get_subjects_list.short_description = 'Subjects'




# class BundleSubscriptionkeys(models.Model):
#     id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
#     details = models.ForeignKey(BundleSubscriptionDetails, on_delete=models.CASCADE, related_name='subscription_keys', blank=True, null=True)
#     key = models.CharField(max_length=255, unique=True, blank=True, null=True)
#     created_at = models.DateTimeField(auto_now_add=True, blank=True, null=True)
#     is_used = models.BooleanField(default=False)

#     def __str__(self):
#         return self.key or ""

class BundleSubscription(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    bundle_subscription_details = models.ForeignKey(BundleSubscriptionDetails, on_delete=models.CASCADE, related_name='bundle_subscriptions')
    student = models.ForeignKey(StudentProfile, on_delete=models.CASCADE, related_name='bundle_subscriptions')
    class_level = models.ForeignKey(ClassLevel, on_delete=models.SET_NULL, blank=True, null=True)
    subjects = models.ManyToManyField(Subject, related_name='bundle_subscriptions')
    start_date = models.DateField(auto_now_add=True, blank=True, null=True)
    end_date = models.DateField( blank=True, null=True)
    is_active = models.BooleanField(default=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    def is_expired(self):
        return timezone.now().date() > self.end_date if self.end_date else False

    def days_remaining(self):
        return (self.end_date - timezone.now().date()).days if self.end_date else 0

    def get_subjects_list(self):
        return ", ".join([subject.name for subject in self.subjects.all()])
    
    get_subjects_list.short_description = 'Subjects'

    def __str__(self):
        return f"{self.student.user.username if self.student else 'N/A'} - {self.bundle_subscription_details.title if self.bundle_subscription_details else 'Bundle'}"

    class Meta:
        ordering = ['-start_date']