from django.contrib import admin
from import_export.admin import ExportMixin, ImportExportMixin
from import_export import resources, fields
from import_export.widgets import ForeignKeyWidget
from .models import (
    AnnualSubscriptionDetails, AnnualSubscriptionkeys, AnnualSubscription,
    SubjectBasedSubscriptionDetails, SubjectBasedSubscriptionkeys, SubjectBasedSubscription,
    BundleSubscriptionDetails, BundleSubscription,
    ClassLevel, Subject, StudentProfile
)

# ==== Resources ====
class AnnualSubscriptionDetailsResource(resources.ModelResource):
    class_level = fields.Field(
        column_name='class_level',
        attribute='class_level',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )
    subject = fields.Field(
        column_name='subject',
        attribute='subject',
        widget=ForeignKeyWidget(Subject, 'name')
    )

    class Meta:
        model = AnnualSubscriptionDetails
        fields = ('id', 'title', 'class_level', 'subject', 'price')


class SubjectBasedSubscriptionDetailsResource(resources.ModelResource):
    class_level = fields.Field(
        column_name='class_level',
        attribute='class_level',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )
    subject = fields.Field(
        column_name='subject',
        attribute='subject',
        widget=ForeignKeyWidget(Subject, 'name')
    )

    class Meta:
        model = SubjectBasedSubscriptionDetails
        fields = ('id', 'title', 'class_level', 'subject', 'price')


class BundleSubscriptionDetailsResource(resources.ModelResource):
    class_level = fields.Field(
        column_name='class_level',
        attribute='class_level',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )

    class Meta:
        model = BundleSubscriptionDetails
        fields = ('id', 'title', 'description', 'class_level', 'price', 'created_at')


class AnnualSubscriptionResource(resources.ModelResource):
    student = fields.Field(
        column_name='student',
        attribute='student',
        widget=ForeignKeyWidget(StudentProfile, 'user__username')
    )
    class_level = fields.Field(
        column_name='class_level',
        attribute='class_level',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )

    class Meta:
        model = AnnualSubscription
        fields = ('id', 'student', 'class_level', 'start_date', 'end_date', 'is_active')


class SubjectBasedSubscriptionResource(resources.ModelResource):
    student = fields.Field(
        column_name='student',
        attribute='student',
        widget=ForeignKeyWidget(StudentProfile, 'user__username')
    )
    class_level = fields.Field(
        column_name='class_level',
        attribute='class_level',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )

    class Meta:
        model = SubjectBasedSubscription
        fields = ('id', 'student', 'class_level', 'start_date', 'end_date', 'is_active')


class BundleSubscriptionResource(resources.ModelResource):
    student = fields.Field(
        column_name='student',
        attribute='student',
        widget=ForeignKeyWidget(StudentProfile, 'user__username')
    )
    class_level = fields.Field(
        column_name='class_level',
        attribute='class_level',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )

    class Meta:
        model = BundleSubscription
        fields = ('id', 'student', 'class_level', 'start_date', 'end_date', 'is_active')


# ==== Inline for Keys ====
class AnnualSubscriptionKeyInline(admin.TabularInline):
    model = AnnualSubscriptionkeys
    extra = 1
    fields = ('key',)
    show_change_link = False


class SubjectSubscriptionKeyInline(admin.TabularInline):
    model = SubjectBasedSubscriptionkeys
    extra = 1
    fields = ('key',)
    show_change_link = False


# ==== Annual Subscription ====
@admin.register(AnnualSubscriptionDetails)
class AnnualSubscriptionDetailsAdmin(ImportExportMixin, admin.ModelAdmin):
    resource_class = AnnualSubscriptionDetailsResource
    list_display = ('title', 'class_level', 'subject', 'price')
    search_fields = ('title', 'class_level__name', 'subject__name')
    list_filter = ('class_level', 'subject')
    ordering = ('title',)
    inlines = [AnnualSubscriptionKeyInline]


@admin.register(AnnualSubscription)
class AnnualSubscriptionAdmin(ImportExportMixin, admin.ModelAdmin):
    resource_class = AnnualSubscriptionResource
    list_display = ('student', 'class_level', 'start_date', 'end_date', 'is_active', 'days_remaining')
    search_fields = ('student__user__username', 'class_level__name')
    list_filter = ('class_level', 'is_active', 'start_date', 'end_date')
    ordering = ('-start_date',)
    readonly_fields = ('days_remaining', 'is_expired')


# ==== Subject-Based Subscription ====
@admin.register(SubjectBasedSubscriptionDetails)
class SubjectBasedSubscriptionDetailsAdmin(ImportExportMixin, admin.ModelAdmin):
    resource_class = SubjectBasedSubscriptionDetailsResource
    list_display = ('title', 'class_level', 'subject', 'price')
    search_fields = ('title', 'class_level__name', 'subject__name')
    list_filter = ('class_level', 'subject')
    ordering = ('title',)
    inlines = [SubjectSubscriptionKeyInline]


@admin.register(SubjectBasedSubscription)
class SubjectBasedSubscriptionAdmin(ImportExportMixin, admin.ModelAdmin):
    resource_class = SubjectBasedSubscriptionResource
    list_display = ('student', 'class_level', 'start_date', 'end_date', 'is_active', 'days_remaining')
    search_fields = ('student__user__username', 'class_level__name')
    list_filter = ('class_level', 'is_active', 'start_date', 'end_date', 'subjects')
    filter_horizontal = ('subjects',)
    ordering = ('-start_date',)
    readonly_fields = ('days_remaining', 'is_expired')


# ==== Bundle Subscription ====
@admin.register(BundleSubscriptionDetails)
class BundleSubscriptionDetailsAdmin(ImportExportMixin, admin.ModelAdmin):
    resource_class = BundleSubscriptionDetailsResource
    list_display = ('title', 'class_level', 'get_subjects_list', 'price', 'created_at')
    search_fields = ('title', 'class_level__name', 'subjects__name')
    list_filter = ('class_level', 'created_at')
    filter_horizontal = ('subjects',)
    ordering = ('-created_at',)

    fieldsets = (
        ('Basic Information', {
            'fields': ('title', 'description', 'class_level', 'price', 'image')
        }),
        ('Subjects', {
            'fields': ('subjects',)
        }),
    )

    def get_subjects_list(self, obj):
        return ", ".join([s.name for s in obj.subjects.all()])
    get_subjects_list.short_description = "Subjects"


@admin.register(BundleSubscription)
class BundleSubscriptionAdmin(ImportExportMixin, admin.ModelAdmin):
    resource_class = BundleSubscriptionResource
    list_display = ('student', 'bundle_subscription_details', 'class_level',
                    'end_date', 'is_active', 'days_remaining_display', 'is_expired_display')
    search_fields = ('student__user__username', 'bundle_subscription_details__title',
                     'class_level__name', 'subjects__name')
    list_filter = ('class_level', 'is_active', 'end_date', 'subjects')
    filter_horizontal = ('subjects',)
    list_per_page = 20
    readonly_fields = ('days_remaining', 'is_expired', 'created_at', 'updated_at', 'start_date')

    fieldsets = (
        ('Subscription Information', {
            'fields': ('bundle_subscription_details', 'student', 'class_level')
        }),
        ('Dates', {
            'fields': ('start_date', 'end_date', 'is_active')
        }),
        ('Subjects', {
            'fields': ('subjects',)
        }),
        ('Metadata', {
            'fields': ('created_at', 'updated_at'),
            'classes': ('collapse',)
        }),
    )

    def days_remaining_display(self, obj):
        return obj.days_remaining()
    days_remaining_display.short_description = 'Days Remaining'
    days_remaining_display.admin_order_field = 'end_date'

    def is_expired_display(self, obj):
        return obj.is_expired()
    is_expired_display.short_description = 'Expired?'
    is_expired_display.boolean = True

    def get_queryset(self, request):
        return super().get_queryset(request).prefetch_related('subjects')
