from django.shortcuts import render, redirect
from django.contrib.auth import login, authenticate, logout
from django.contrib import messages
from django.contrib.auth.decorators import login_required
from django.core.files.storage import FileSystemStorage
from django.conf import settings
from django.contrib.auth import get_backends
from app.models import ClassLevel, User, StudentProfile, Question, Exam, ExamResult


# =============================================================================
# ADMIN DASHBOARD CALLBACK FOR DJANGO UNFOLD
# =============================================================================
def dashboard_callback(request, context):
    """
    Custom dashboard callback for Django Unfold admin.
    Adds statistics and recent activity to the admin dashboard.
    """
    from subscription.models import BundleSubscription, AnnualSubscription
    
    context.update({
        "stats": [
            {
                "title": "Total Students",
                "value": StudentProfile.objects.count(),
                "icon": "school",
                "color": "primary",
            },
            {
                "title": "Total Questions",
                "value": Question.objects.count(),
                "icon": "quiz",
                "color": "success",
            },
            {
                "title": "Exams Taken",
                "value": ExamResult.objects.count(),
                "icon": "assignment",
                "color": "info",
            },
            {
                "title": "Active Subscriptions",
                "value": BundleSubscription.objects.filter(is_active=True).count() + 
                         AnnualSubscription.objects.filter(is_active=True).count(),
                "icon": "subscriptions",
                "color": "warning",
            },
        ],
        "recent_exams": ExamResult.objects.select_related('exam', 'exam__student', 'exam__student__user').order_by('-completed_at')[:5],
    })
    return context





def signup_user(request):
    
    classes = ClassLevel.objects.all()
    return render(request, 'auth/signup.html', {
        'classes': classes,
        'GOOGLE_OAUTH_CLIENT_ID': settings.SOCIAL_AUTH_GOOGLE_OAUTH2_KEY,
        # 'FACEBOOK_APP_ID': settings.FACEBOOK_APP_ID
    })



def save_signup_user(request):
    if request.method == 'POST':
        # Get form data
        first_name = request.POST.get('first_name')
        last_name = request.POST.get('last_name')
        phone_number = request.POST.get('phone_number')
        email = request.POST.get('email')
        password1 = request.POST.get('password1')
        password2 = request.POST.get('password2')
        school_name = request.POST.get('school_name')
        current_class_id = request.POST.get('current_class')
        profile_picture = request.FILES.get('profile_picture')

        # Validation
        if not all([first_name, last_name, phone_number, email, password1, password2]):
            messages.error(request, 'Please fill all required fields')
            return redirect('signup_user')

        if password1 != password2:
            messages.error(request, 'Passwords do not match')
            return redirect('signup_user')

        if User.objects.filter(email=email).exists():
            messages.error(request, 'Email already exists')
            return redirect('signup_user')

        # Create user
        try:
            user = User.objects.create_user(
                username=email,
                email=email,
                password=password1,
                first_name=first_name,
                last_name=last_name,
                phone_number=phone_number
            )

            # Handle profile picture
            if profile_picture:
                fs = FileSystemStorage()
                filename = fs.save(f'profile_pics/{user.id}_{profile_picture.name}', profile_picture)
                user.profile_picture = filename
                user.save()
                print("Picture added\n\n")  

            # Create student profile
            current_class = ClassLevel.objects.get(id=current_class_id) if current_class_id else None
            StudentProfile.objects.create(
                user=user,
                school_name=school_name,
                current_class=current_class
            )

            # Important fix: Specify backend for login because of multiple authentication backends
            backend_str = 'django.contrib.auth.backends.ModelBackend'
            user.backend = backend_str
            request.session['_auth_user_backend'] = backend_str

            # Log the user in
            login(request, user)
            messages.success(request, 'Account created successfully!')
            return redirect('app:dashboard')

        except Exception as e:
            messages.error(request, f'Error creating account: {str(e)}')
            return redirect('signup_user')
    return render(request, 'auth/signup.html')



def login_user(request):
    if request.method == 'POST':
        email = request.POST.get('email')
        password = request.POST.get('password')
        user = authenticate(request, username=email, password=password)
        
        if user is not None:
            user.backend = 'django.contrib.auth.backends.ModelBackend'
            login(request, user)
            return redirect('app:dashboard')
        else:
            messages.error(request, 'Invalid email or password.')
    
    return render(request, 'auth/login.html', {
        'GOOGLE_OAUTH_CLIENT_ID': settings.SOCIAL_AUTH_GOOGLE_OAUTH2_KEY,
        # 'FACEBOOK_APP_ID': settings.FACEBOOK_APP_ID
    })




def logout_user(request):
    logout(request)
    return redirect('login_user')



def create_student_profile(strategy, details, backend, user, *args, **kwargs):
    if not hasattr(user, 'studentprofile'):
        StudentProfile.objects.create(user=user)