from django.db import models
from django.contrib.auth import get_user_model
from app.models import ClassLevel, Subject, Chapter
import uuid

User = get_user_model()



class Bookmark(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='user_bookmarks')
    doubt = models.ForeignKey('Doubt', on_delete=models.CASCADE, related_name='doubt_bookmarks')
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        unique_together = ('user', 'doubt')
        db_table = 'doubts_bookmark'

    def __str__(self):
        return f"{self.user.username} bookmarked {self.doubt.id}"
    

class Doubt(models.Model):
    POST_TYPES = (
        ('text', 'Text Post'),
        ('image', 'Image Post'),
        ('document', 'Document Post'),
        ('link', 'Link Post'),
    )
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='doubts')
    post_type = models.CharField(max_length=10, choices=POST_TYPES)
    class_level = models.ForeignKey(ClassLevel, on_delete=models.CASCADE)
    subject = models.ForeignKey(Subject, on_delete=models.CASCADE)
    chapter = models.ForeignKey(Chapter, on_delete=models.SET_NULL, null=True, blank=True)
    title = models.CharField(max_length=200)
    content = models.TextField(blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    likes = models.ManyToManyField(User, related_name='liked_doubts', blank=True)
    bookmarks = models.ManyToManyField(User, through=Bookmark, related_name='bookmarked_doubts', blank=True)

    class Meta:
        ordering = ['-created_at']
        db_table = 'doubts_doubt'

    def __str__(self):
        return f"{self.title} by {self.user.username}"

    def total_likes(self):
        return self.likes.count()

    def total_comments(self):
        return self.comments.count()

    def total_bookmarks(self):
        return self.bookmarks.count()
    

    

class DoubtImage(models.Model):
    doubt = models.ForeignKey(Doubt, on_delete=models.CASCADE, related_name='images')
    image = models.ImageField(upload_to='doubts/images/')
    caption = models.CharField(max_length=200, blank=True, null=True)

    class Meta:
        db_table = 'doubts_doubtimage'

    def __str__(self):
        return f"Image for {self.doubt.title}"

class DoubtDocument(models.Model):
    doubt = models.ForeignKey(Doubt, on_delete=models.CASCADE, related_name='documents')
    document = models.FileField(upload_to='doubts/documents/')
    description = models.CharField(max_length=200, blank=True, null=True)

    class Meta:
        db_table = 'doubts_doubtdocument'

    def __str__(self):
        return f"Document for {self.doubt.title}"

class DoubtLink(models.Model):
    doubt = models.ForeignKey(Doubt, on_delete=models.CASCADE, related_name='links')
    url = models.URLField()
    description = models.CharField(max_length=200, blank=True, null=True)

    class Meta:
        db_table = 'doubts_doubtlink'

    def __str__(self):
        return f"Link for {self.doubt.title}"

class Comment(models.Model):
    doubt = models.ForeignKey(Doubt, on_delete=models.CASCADE, related_name='comments')
    user = models.ForeignKey(User, on_delete=models.CASCADE)
    content = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    class Meta:
        ordering = ['created_at']
        db_table = 'doubts_comment'

    def __str__(self):
        return f"Comment by {self.user.username} on {self.doubt.title}"

class Notification(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='notifications')
    doubt = models.ForeignKey(Doubt, on_delete=models.CASCADE, null=True, blank=True)
    comment = models.ForeignKey(Comment, on_delete=models.CASCADE, null=True, blank=True)
    is_read = models.BooleanField(default=False)
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        ordering = ['-created_at']
        db_table = 'doubts_notification'

    def __str__(self):
        if self.comment:
            return f"New comment on your doubt by {self.comment.user.username}"
        return "New notification"
