import uuid
from django.db import models
from django.contrib.auth.models import AbstractUser, Group, Permission
from phonenumber_field.modelfields import PhoneNumberField
from django.utils.translation import gettext_lazy as _
from django.core.validators import MinValueValidator, MaxValueValidator
from django_ckeditor_5.fields import CKEditor5Field




class User(AbstractUser):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    phone_number = PhoneNumberField(_('phone number'), blank=True, null=True)
    profile_picture = models.ImageField(upload_to='profile_pics/', blank=True, null=True)

    groups = models.ManyToManyField(
        Group,
        related_name='custom_user_set',
        blank=True,
        help_text='The groups this user belongs to.',
        verbose_name='groups',
    )
    user_permissions = models.ManyToManyField(
        Permission,
        related_name='custom_user_permissions',
        blank=True,
        help_text='Specific permissions for this user.',
        verbose_name='user permissions',
    )


class ClassLevel(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    name = models.CharField(max_length=50)
    order = models.PositiveIntegerField()
    image = models.ImageField(upload_to='class_images/', blank=True, null=True)

    def __str__(self):
        return self.name


class Subject(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    name = models.CharField(max_length=100)
    class_level = models.ForeignKey(ClassLevel, on_delete=models.CASCADE, related_name='subjects')
    image = models.ImageField(upload_to='subject_images/', blank=True, null=True)

    def __str__(self):
        return f"{self.name} - {self.class_level.name}"


class Chapter(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    class_level = models.ForeignKey(ClassLevel, on_delete=models.CASCADE, related_name='chapters_class', blank=True, null=True)
    subject = models.ForeignKey(Subject, on_delete=models.CASCADE, related_name='chapters')
    title = models.CharField(max_length=255)

    def __str__(self):
        return f"{self.title} ({self.subject.name})"


class ExamCategory(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    name = models.CharField(max_length=100)

    def __str__(self):
        return self.name


class Question(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    QUESTION_TYPES = [
        ('MCQ', 'Multiple Choice Question'),
        ('TF', 'True/False'),
        ('FB', 'Fill in the Blanks'),
    ]

    text = models.CharField(max_length=255, help_text="প্রশ্নের টেক্সট", default="", blank=True, null=True)
    question_type = models.CharField(max_length=5, choices=QUESTION_TYPES, default='MCQ')
    subject = models.ForeignKey(Subject, on_delete=models.CASCADE, related_name='questions')
    chapter = models.ForeignKey(Chapter, on_delete=models.SET_NULL, null=True, blank=True, related_name='questions')
    exam_category = models.ForeignKey(ExamCategory, on_delete=models.SET_NULL, null=True, blank=True)
    difficulty = models.PositiveSmallIntegerField(default=1, validators=[MinValueValidator(1), MaxValueValidator(5)])

    def __str__(self):
        return f"Q: {self.text[:50]}..."


class Choice(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    question = models.ForeignKey(Question, on_delete=models.CASCADE, related_name='choices')
    text = models.CharField(max_length=255)
    is_correct = models.BooleanField(default=False)
    explanation = CKEditor5Field(help_text="কারেক্ট উত্তর কেন সঠিক তার ব্যাখ্যা", blank=True, null=True)

    def __str__(self):
        return self.text


class StudentProfile(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    user = models.OneToOneField(User, on_delete=models.CASCADE)
    school_name = models.CharField(max_length=255, blank=True, null=True)
    current_class = models.ForeignKey(ClassLevel, on_delete=models.SET_NULL, null=True)

    def __str__(self):
        return self.user.username


class Exam(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    title = models.CharField(max_length=255)
    student = models.ForeignKey(StudentProfile, on_delete=models.CASCADE, related_name='exams')
    class_level = models.ForeignKey(ClassLevel, on_delete=models.SET_NULL, null=True)
    subject = models.ForeignKey(Subject, on_delete=models.SET_NULL, null=True)
    exam_category = models.ForeignKey(ExamCategory, on_delete=models.SET_NULL, null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    duration_minutes = models.PositiveIntegerField(default=30)
    is_custom = models.BooleanField(default=False)

    def __str__(self):
        return f"{self.title} ({self.student.user.username})"


class ExamQuestion(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    exam = models.ForeignKey(Exam, on_delete=models.CASCADE, related_name='exam_questions')
    question = models.ForeignKey(Question, on_delete=models.CASCADE)

    class Meta:
        # Ensures that a question is only added once per exam
        unique_together = ('exam', 'question')


class ExamResult(models.Model):
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    exam = models.OneToOneField(Exam, on_delete=models.CASCADE, related_name='result')
    score = models.DecimalField(max_digits=5, decimal_places=2)
    correct_answers = models.PositiveIntegerField()
    wrong_answers = models.PositiveIntegerField()
    time_taken_seconds = models.PositiveIntegerField()
    completed_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return f"Result for {self.exam.title} by {self.exam.student.user.username}"

class ExamResultDetails(models.Model):
    """
    Stores details for each question in an exam result, including the user's selected
    answer and the correct answer.
    """
    id = models.UUIDField(primary_key=True, default=uuid.uuid4, editable=False)
    exam_result = models.ForeignKey(ExamResult, on_delete=models.CASCADE, related_name='details')
    exam_question = models.ForeignKey(ExamQuestion, on_delete=models.CASCADE)
    selected_choice = models.ForeignKey(Choice, on_delete=models.SET_NULL, null=True, blank=True, related_name='selected_by_results')
    correct_choice = models.ForeignKey(Choice, on_delete=models.SET_NULL, null=True, blank=True, related_name='correct_for_results')
    is_correct = models.BooleanField(default=False)

    def save(self, *args, **kwargs):
        # Ensure correct_choice is set if not provided
        if not self.correct_choice:
            self.correct_choice = self.exam_question.question.choices.filter(is_correct=True).first()
        super().save(*args, **kwargs)

    def __str__(self):
        return f"Detail for {self.exam_result.exam.title} - Q: {self.exam_question.question.text[:30]}"