import csv
import requests
from io import StringIO
from django.core.management.base import BaseCommand
from app.models import Question, Choice, Subject, Chapter, ExamCategory

class Command(BaseCommand):
    help = 'Import questions and choices from a Google Drive CSV link'

    def handle(self, *args, **kwargs):
        google_drive_url = 'https://drive.usercontent.google.com/u/0/uc?id=1Ni5u7xExGBzuUP3m674c1XiJQlWl4g3u&export=download'  # Your Google Drive URL
        response = requests.get(google_drive_url)

        if response.status_code == 200:
            file_content = response.text  # Get the CSV file content from Google Drive
            csv_file = StringIO(file_content)  # Use StringIO to simulate a file object
            reader = csv.DictReader(csv_file)

            for row in reader:
                # Extract data from CSV
                question_text = row['Question Text']
                question_type = row['Question Type']
                subject_name = row['Subject']
                chapter_title = row['Chapter']
                exam_category_name = row['Exam Category']
                difficulty = row['Difficulty']
                
                # Get or create related entities like Subject, Chapter, and ExamCategory
                subject, created = Subject.objects.get_or_create(name=subject_name)
                chapter, created = Chapter.objects.get_or_create(title=chapter_title, subject=subject)
                exam_category, created = ExamCategory.objects.get_or_create(name=exam_category_name)
                
                # Create the Question
                question = Question.objects.create(
                    text=question_text,
                    question_type=question_type,
                    subject=subject,
                    chapter=chapter,
                    exam_category=exam_category,
                    difficulty=difficulty,
                )

                # Extract Choices from the CSV
                choice_1 = row['Choice 1']
                choice_2 = row['Choice 2']
                choice_3 = row['Choice 3']
                choice_4 = row['Choice 4']
                correct_choice = row['Correct Choice']
                explanation = row['Correct Answer Explanation']

                # Add choices to the question
                for choice_text, is_correct in [(choice_1, correct_choice == '1'), 
                                                (choice_2, correct_choice == '2'),
                                                (choice_3, correct_choice == '3'),
                                                (choice_4, correct_choice == '4')]:
                    Choice.objects.create(
                        question=question,
                        text=choice_text,
                        is_correct=is_correct,
                        explanation=explanation if is_correct else '',
                    )
                    
            self.stdout.write(self.style.SUCCESS('Data successfully imported'))
        else:
            self.stdout.write(self.style.ERROR('Failed to download file from Google Drive'))
