import os
import django
import random
from faker import Faker
from datetime import timedelta, date

# Set up Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'passMCQ.settings')  # Change this to your project name
django.setup()

from app.models import (
    User, StudentProfile, ClassLevel, Subject, Chapter, ExamCategory,
    Question, Choice, AnnualSubscription, SubjectBasedSubscription,
    BundleSubscription, UserBundleSubscription, Exam, ExamQuestion, ExamResult
)

fake = Faker()

# --- Helper Functions ---

def create_class_levels(n=3):
    class_levels = []
    for i in range(n):
        cl = ClassLevel.objects.create(
            name=f"Class {i+1}",
            order=i+1
        )
        class_levels.append(cl)
    return class_levels

def create_subjects(class_levels, n_per_class=3):
    subjects = []
    for cl in class_levels:
        for _ in range(n_per_class):
            subj = Subject.objects.create(
                name=fake.word().capitalize() + " Studies",
                class_level=cl
            )
            subjects.append(subj)
    return subjects

def create_chapters(subjects, n_per_subject=3):
    for subj in subjects:
        for _ in range(n_per_subject):
            Chapter.objects.create(
                subject=subj,
                title=fake.sentence(nb_words=4)
            )

def create_exam_categories(n=3):
    for _ in range(n):
        ExamCategory.objects.create(name=fake.word().capitalize() + " Exams")

def create_students(n=10):
    students = []
    for _ in range(n):
        user = User.objects.create_user(
            username=fake.unique.user_name(),
            email=fake.unique.email(),
            password='password123'
        )
        profile = StudentProfile.objects.create(
            user=user,
            school_name=fake.company(),
            current_class=random.choice(ClassLevel.objects.all())
        )
        students.append(profile)
    return students

def create_questions(subjects, n=5):
    for subj in subjects:
        for _ in range(n):
            q = Question.objects.create(
                text=fake.sentence(nb_words=10),
                question_type=random.choice(['MCQ', 'TF', 'FB']),
                subject=subj,
                chapter=random.choice(list(subj.chapters.all())) if subj.chapters.exists() else None,
                exam_category=random.choice(ExamCategory.objects.all()),
                difficulty=random.randint(1, 5)
            )
            # Create choices
            for i in range(4):
                Choice.objects.create(
                    question=q,
                    text=fake.word().capitalize(),
                    is_correct=(i == 0)
                )

def create_subscriptions(students):
    for student in students:
        class_level = student.current_class
        subjects = Subject.objects.filter(class_level=class_level)
        end_date = date.today() + timedelta(days=365)

        # Annual
        AnnualSubscription.objects.create(
            student=student,
            class_level=class_level,
            end_date=end_date
        )

        # Subject-based
        sub_based = SubjectBasedSubscription.objects.create(
            student=student,
            class_level=class_level,
            end_date=end_date
        )
        sub_based.subjects.set(random.sample(list(subjects), min(2, subjects.count())))

        # Bundle
        bundle = BundleSubscription.objects.create(
            name=f"{class_level.name} Premium",
            class_level=class_level,
            price=random.randint(500, 2000),
            duration_days=365
        )
        bundle.subjects.set(subjects)

        UserBundleSubscription.objects.create(
            student=student,
            bundle=bundle,
            end_date=end_date
        )

def create_exams(students):
    for student in students:
        class_level = student.current_class
        subjects = Subject.objects.filter(class_level=class_level)
        subject = random.choice(subjects)
        exam = Exam.objects.create(
            title=f"{subject.name} Test",
            student=student,
            class_level=class_level,
            subject=subject,
            exam_category=random.choice(ExamCategory.objects.all()),
            duration_minutes=30
        )

        questions = Question.objects.filter(subject=subject)[:5]
        correct = 0
        for q in questions:
            ExamQuestion.objects.create(exam=exam, question=q)
            correct += 1  # Simulate correct answers

        ExamResult.objects.create(
            exam=exam,
            score=round((correct / len(questions)) * 100, 2),
            correct_answers=correct,
            wrong_answers=len(questions) - correct,
            time_taken_seconds=random.randint(300, 1200)
        )

# --- Execution Pipeline ---

def run():
    print("Generating fake data...")
    class_levels = create_class_levels()
    subjects = create_subjects(class_levels)
    create_chapters(subjects)
    create_exam_categories()
    students = create_students()
    create_questions(subjects)
    create_subscriptions(students)
    create_exams(students)
    print("✅ Done generating data.")

if __name__ == '__main__':
    run()
