from django.contrib import admin
from import_export import resources, fields
from import_export.admin import ImportExportModelAdmin
from import_export.widgets import ForeignKeyWidget
from .models import (
    User, ClassLevel, Subject, Chapter, ExamCategory,
    Question, Choice, StudentProfile,
    Exam, ExamQuestion, ExamResult, ExamResultDetails
)

# === ClassLevel Admin ===
class ClassLevelResource(resources.ModelResource):
    class Meta:
        model = ClassLevel
        fields = ('id', 'name', 'order')
        import_id_fields = ['id']

@admin.register(ClassLevel)
class ClassLevelAdmin(ImportExportModelAdmin):
    resource_class = ClassLevelResource
    list_display = ('name', 'order')
    search_fields = ('name',)
    list_filter = ('order',)


# === Inlines for Admin views ===
class ChoiceInline(admin.TabularInline):
    model = Choice
    extra = 4  # Show 4 empty choice forms by default

class ExamQuestionInline(admin.TabularInline):
    model = ExamQuestion
    extra = 1  # Show 1 empty exam question form by default

class ExamResultDetailsInline(admin.TabularInline):
    model = ExamResultDetails
    extra = 0
    readonly_fields = ('exam_question', 'selected_choice', 'correct_choice', 'is_correct')
    can_delete = False


# === Question Admin with Import/Export ===
class QuestionResource(resources.ModelResource):
    id = fields.Field(column_name='Question ID', attribute='id')
    text = fields.Field(column_name='Question Text', attribute='text')
    question_type = fields.Field(column_name='Question Type', attribute='question_type')

    subject = fields.Field(
        column_name='Subject',
        attribute='subject',
        widget=ForeignKeyWidget(Subject, 'name')
    )
    chapter = fields.Field(
        column_name='Chapter',
        attribute='chapter',
        widget=ForeignKeyWidget(Chapter, 'title')
    )
    exam_category = fields.Field(
        column_name='Exam Category',
        attribute='exam_category',
        widget=ForeignKeyWidget(ExamCategory, 'name')
    )
    difficulty = fields.Field(column_name='Difficulty', attribute='difficulty')

    choice_1 = fields.Field(column_name='Choice 1', attribute='choice_1')
    choice_2 = fields.Field(column_name='Choice 2', attribute='choice_2')
    choice_3 = fields.Field(column_name='Choice 3', attribute='choice_3')
    choice_4 = fields.Field(column_name='Choice 4', attribute='choice_4')
    correct_choice = fields.Field(column_name='Correct Choice', attribute='correct_choice')
    explanation = fields.Field(column_name='Correct Answer Explanation', attribute='explanation')

    class Meta:
        model = Question
        import_id_fields = ['id']
        fields = (
            'id', 'text', 'question_type', 'subject', 'chapter',
            'exam_category', 'difficulty',
            'choice_1', 'choice_2', 'choice_3', 'choice_4',
            'correct_choice', 'explanation'
        )
        skip_unchanged = True
        report_skipped = True

    def after_save_instance(self, instance, row, **kwargs):
        """
        This method is called after a Question instance is saved.
        We use it to create the related Choice objects from the CSV data.
        """
        dry_run = kwargs.get('dry_run', False)
        if dry_run:
            return

        # Clear previous choices to avoid duplicating entries
        instance.choices.all().delete()

        # Retrieve choice data from CSV row and create Choice objects
        choices = [
            str(row.get("Choice 1", "")).strip(),
            str(row.get("Choice 2", "")).strip(),
            str(row.get("Choice 3", "")).strip(),
            str(row.get("Choice 4", "")).strip(),
        ]
        correct = str(row.get("Correct Choice", "")).strip()
        explanation = str(row.get("Correct Answer Explanation", "")).strip()

        # Create choices and assign explanation to the correct one
        for choice_text in choices:
            if choice_text:
                is_correct = (choice_text == correct)
                Choice.objects.create(
                    question=instance,
                    text=choice_text,
                    is_correct=is_correct,
                    explanation=explanation if is_correct else ""
                )


@admin.register(Question)
class QuestionAdmin(ImportExportModelAdmin):
    resource_class = QuestionResource
    list_display = ('text', 'question_type', 'subject', 'chapter', 'exam_category', 'difficulty')
    list_filter = ('question_type', 'subject__class_level', 'subject', 'chapter', 'exam_category', 'difficulty')
    search_fields = ('text', 'subject__name', 'chapter__title')
    inlines = [ChoiceInline]


# === Subject Admin ===
class SubjectResource(resources.ModelResource):
    class_level = fields.Field(
        column_name='class_level',
        attribute='class_level',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )

    class Meta:
        model = Subject
        fields = ('id', 'name', 'class_level')
        import_id_fields = ['id']

@admin.register(Subject)
class SubjectAdmin(ImportExportModelAdmin):
    resource_class = SubjectResource
    list_display = ('name', 'class_level')
    list_filter = ('class_level',)
    search_fields = ('name',)


# === Chapter Admin ===
class ChapterResource(resources.ModelResource):
    subject = fields.Field(
        column_name='subject',
        attribute='subject',
        widget=ForeignKeyWidget(Subject, 'name')
    )
    class_level = fields.Field(
        column_name='class_level',
        attribute='class_level',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )

    class Meta:
        model = Chapter
        fields = ('id', 'title', 'subject', 'class_level')
        import_id_fields = ['id']

@admin.register(Chapter)
class ChapterAdmin(ImportExportModelAdmin):
    resource_class = ChapterResource
    list_display = ('title', 'subject', 'class_level')
    list_filter = ('subject__class_level', 'subject')
    search_fields = ('title',)

    # Filtering ClassLevel based on the selected Subject
    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == 'class_level':
            subject = request.GET.get('subject')  # Get selected subject from the URL query
            if subject:
                kwargs['queryset'] = ClassLevel.objects.filter(subject__id=subject)
        return super().formfield_for_foreignkey(db_field, request, **kwargs)


# === ExamCategory Admin ===
class ExamCategoryResource(resources.ModelResource):
    class Meta:
        model = ExamCategory
        fields = ('id', 'name')
        import_id_fields = ['id']

@admin.register(ExamCategory)
class ExamCategoryAdmin(ImportExportModelAdmin):
    resource_class = ExamCategoryResource
    list_display = ('name',)
    search_fields = ('name',)


# === StudentProfile Admin ===
class StudentProfileResource(resources.ModelResource):
    user = fields.Field(
        column_name='user',
        attribute='user',
        widget=ForeignKeyWidget(User, 'username')
    )
    current_class = fields.Field(
        column_name='current_class',
        attribute='current_class',
        widget=ForeignKeyWidget(ClassLevel, 'name')
    )

    class Meta:
        model = StudentProfile
        fields = ('id', 'user', 'school_name', 'current_class')
        import_id_fields = ['id']

@admin.register(StudentProfile)
class StudentProfileAdmin(ImportExportModelAdmin):
    resource_class = StudentProfileResource
    list_display = ('user', 'school_name', 'current_class')
    search_fields = ('user__username', 'school_name')
    list_filter = ('current_class',)


# === Exam Admin ===
@admin.register(Exam)
class ExamAdmin(admin.ModelAdmin):
    list_display = (
        'title', 'student', 'class_level', 'subject', 'exam_category',
        'created_at', 'duration_minutes', 'is_custom'
    )
    list_filter = ('class_level', 'subject', 'exam_category', 'is_custom')
    search_fields = ('title', 'student__user__username')
    inlines = [ExamQuestionInline]
    fieldsets = (
        ("Basic Info", {
            "fields": ("title", "student", "class_level", "subject", "exam_category"),
        }),
        ("Timing & Settings", {
            "fields": ("duration_minutes", "is_custom"),
        }),
    )


# === ExamResult Admin ===
@admin.register(ExamResult)
class ExamResultAdmin(admin.ModelAdmin):
    list_display = (
        'exam', 'score', 'correct_answers', 'wrong_answers',
        'time_taken_seconds', 'completed_at'
    )
    search_fields = ('exam__title', 'exam__student__user__username')
    list_filter = ('exam__student__current_class', 'score', 'completed_at')
    inlines = [ExamResultDetailsInline]
    readonly_fields = ('exam', 'score', 'correct_answers', 'wrong_answers', 'time_taken_seconds', 'completed_at')


# Register other models that don't need special admin classes
admin.site.register(User)
admin.site.register(ExamQuestion)
admin.site.register(Choice)
admin.site.register(ExamResultDetails)
